<?php
/**
 * Elementor Widget
 * @package payone
 * @since 1.0.0
 */

namespace Elementor;

class Payone_Team_List_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'payone-team-list-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Team List', 'payone-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['payone_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'slider_settings_section',
            [
                'label' => esc_html__('Slider Settings', 'payone-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control('team_grid', [
            'label' => esc_html__('team Grid', 'payone-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'col-lg-3' => esc_html__('col-lg-2', 'payone-core'),
                'col-lg-3' => esc_html__('col-lg-3', 'payone-core'),
                'col-lg-4' => esc_html__('col-lg-4', 'payone-core'),
                'col-lg-6' => esc_html__('col-lg-6', 'payone-core'),
                'col-lg-12' => esc_html__('col-lg-12', 'payone-core'),
            ),
            'default' => 'col-lg-3',
            'description' => esc_html__('Select team Grid', 'payone-core')
        ]);

        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'payone-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many course you want in masonry , enter -1 for unlimited course.')
        ]);
        $this->add_control('category', [
            'label' => esc_html__('Category', 'payone-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => payone()->get_terms_names('team-cat', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'payone-core'),
        ]);
        $this->add_control('order', [
            'label' => esc_html__('Order', 'payone-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'payone-core'),
                'DESC' => esc_html__('Descending', 'payone-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'payone-core')
        ]);
        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'payone-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'payone-core'),
                'title' => esc_html__('Title', 'payone-core'),
                'date' => esc_html__('Date', 'payone-core'),
                'rand' => esc_html__('Random', 'payone-core'),
                'comment_count' => esc_html__('Most Comments', 'payone-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'payone-core')
        ]);
        $this->end_controls_section();

        $this->start_controls_section(
            'team_member_styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'payone-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control('team_bg_color', [
            'label' => __('team_bg Color', 'payone-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .bg-whitex" => "background-color: {{VALUE}} !important"
            ]
        ]);
        $this->end_controls_section();

    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];
        //setup query
        $args = array(
            'post_type' => 'team',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'team-cat',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
        ?>

        <div class="team-area-1 position-relative z-1">
            <div class="row gy-4">
                <?php
                while ($post_data->have_posts()) : $post_data->the_post();
                    $post_id = get_the_ID();
                    $img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
                    $img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'full', false) : '';
                    $img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
                    $img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
                    $team_single_meta_data = get_post_meta(get_the_ID(), 'payone_team_options', true);
                    
                    $social_icons = isset($team_single_meta_data['social-icons']) && !empty($team_single_meta_data['social-icons']) ? $team_single_meta_data['social-icons'] : '';
                    // icon
                    $team_icon = isset($team_single_meta_data['team_icon']) && !empty($team_single_meta_data['team_icon']) ? $team_single_meta_data['team_icon'] : '';
                    ?>

                    <div class="<?php echo esc_attr($settings['team_grid']); ?> col-md-6 col-sm-6">
                        <div class="group-item group" data-aos="fade-up" data-aos-duration="600">
                            <div class="tw-h-380-px position-relative">
                                <div class="bg-white bg-whitex group-hover-bg-base-two-600 tw-rounded-xl border-neutral-300 hover-border-main-600 position-absolute w-100 h-100 top-0 start-0 text-center tw-visible tw-opacity-1 tw-duration-500">
                                    <img src="<?php echo esc_url($img_url) ?>" alt="Image" class="tw-mt-10">
                                </div>
                                <div class="position-absolute tw-start-0 bottom-0 tw-mb-6 tw-ms-6">
                                    <span class="fw-semibold tw-mb-1 tw-text-lg text-dark-600 d-block tw-duration-500">
                                        <a class="group-hover-text-white" href="<?php the_permalink() ?>"><?php echo esc_html(get_the_title($post_id)) ?></a>
                                    </span>
                                    <span class="fw-normal tw-text-4 text-dark-500 tw-duration-500 group-hover-text-white">
                                        <?php echo $team_single_meta_data['designation'] ?>
                                    </span>
                                </div>
                                <a href="<?php the_permalink() ?>" class="d-flex align-items-center justify-content-center tw-w-13 tw-h-13 bg-white tw-rounded-xl tw-text-6 position-absolute tw-end-0 bottom-0 z-1 group-hover-bg-base-two-600 tw-duration-500 group-hover-text-white">
                                    <i class="ph ph-arrow-up-right"></i>
                                </a>
                                <span class="our-team-member tw-h-72-px tw-w-72-px bg-neutral-10 position-absolute tw-end-0 bottom-0 border-top-left-24-px"></span>
                            </div>
                        </div>
                    </div>
                    
                <?php endwhile; ?>
            </div>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Payone_Team_List_Widget());