<?php
/**
 * @package WordPress
 * @subpackage Jobster
 */

if (!function_exists('jobster_send_job_alerts')): 
    function jobster_send_job_alerts($post_id, $post, $update, $post_before) {
        if ($post->post_status === 'publish' 
            && $post->post_type === 'job'
            && isset($post_before) 
            && $post_before->post_status !== 'publish'
        ) {
            $jobs_settings = get_option('jobster_jobs_settings');
            if (isset($jobs_settings['jobster_job_alert_field']) 
                && $jobs_settings['jobster_job_alert_field'] == '1'
            ) {
                $location = wp_get_post_terms($post->ID, 'job_location');
                $location_id = $location ? $location[0]->term_id : '';
                $category = wp_get_post_terms($post->ID, 'job_category');
                $category_id = $category ? $category[0]->term_id : '';
                $type = wp_get_post_terms($post->ID, 'job_type');
                $type_id = $type ? $type[0]->term_id : '';
                $level = wp_get_post_terms($post->ID, 'job_level');
                $level_id = $level ? $level[0]->term_id : '';

                $job_data = array(
                    'id' => $post->ID,
                    'location' => $location_id,
                    'category' => $category_id,
                    'type' => $type_id,
                    'level' => $level_id,
                );

                $candidates = jobster_get_job_alerts_candidates($job_data);

                foreach ($candidates->posts as $candidate) {
                    jobster_send_jobs_alerts_email($job_data, $candidate);
                }
            }
        }
    }
endif;
add_action('wp_after_insert_post', 'jobster_send_job_alerts', 10, 4);

if (!function_exists('jobster_get_job_alerts_candidates')): 
    function jobster_get_job_alerts_candidates($job_data) {
        $criteria = array('relation' => 'AND');

        if (!empty($job_data['location'])) {
            array_push($criteria, array(
                'relation' => 'OR',
                array(
                    'key' => 'candidate_job_alerts_location',
                    'value' => '"' . $job_data['location'] . '*"',
                    'compare' => 'LIKE'
                ),
                array(
                    'key' => 'candidate_job_alerts_location',
                    'value' => 'a:0:{}',
                    'compare' => '=='
                ),
            ));
        }

        if (!empty($job_data['category'])) {
            array_push($criteria, array(
                'relation' => 'OR',
                array(
                    'key' => 'candidate_job_alerts_category',
                    'value' => '"' . $job_data['category'] . '"',
                    'compare' => 'LIKE'
                ),
                array(
                    'key' => 'candidate_job_alerts_category',
                    'value' => 'a:0:{}',
                    'compare' => '=='
                ),
            ));
        }

        if (!empty($job_data['level'])) {
            array_push($criteria, array(
                'relation' => 'OR',
                array(
                    'key' => 'candidate_job_alerts_level',
                    'value' => '"' . $job_data['level'] . '"',
                    'compare' => 'LIKE'
                ),
                array(
                    'key' => 'candidate_job_alerts_level',
                    'value' => 'a:0:{}',
                    'compare' => '=='
                ),
            ));
        }

        if (!empty($job_data['type'])) {
            array_push($criteria, array(
                'relation' => 'OR',
                array(
                    'key' => 'candidate_job_alerts_type',
                    'value' => '"' . $job_data['type'] . '"',
                    'compare' => 'LIKE'
                ),
                array(
                    'key' => 'candidate_job_alerts_type',
                    'value' => 'a:0:{}',
                    'compare' => '=='
                ),
            ));
        }

        $args = array(
            'posts_per_page' => -1,
            'post_type'      => 'candidate',
            'post_status'    => 'publish',
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key' => 'candidate_job_alerts',
                    'value' => '1'
                ),
                $criteria
            ),
        );

        $query = new WP_Query($args);
        wp_reset_postdata();

        return $query;
    }
endif;

if (!function_exists('jobster_send_jobs_alerts_email')): 
    function jobster_send_jobs_alerts_email($job_data, $candidate) {
        $address = get_post_meta($candidate->ID, 'candidate_email', true);
        if (empty($address)) return;

        $job_title = get_the_title($job_data['id']);
        $job_url = get_the_permalink($job_data['id']);
        $candidate_name = $candidate->post_title;

        $job_company_id = get_post_meta($job_data['id'], 'job_company', true);
        $job_anonymous = get_post_meta($job_data['id'], 'job_anonymous', true);

        $subject = sprintf( __('New Job Alert: [%s]', 'jobster-plugin'), $job_title );
        if (!empty($job_company_id)) {
            $company_name = get_the_title($job_company_id);
            $subject = sprintf(
                __('New Job Alert: [%s] at [%s]', 'jobster-plugin'),
                $job_title,
                $company_name
            );
        }

        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_option('blogname') . '<' . get_option('admin_email') . '>',
            'Reply-To: ' . get_option('blogname') . '<' . get_option('admin_email') . '>'
        );

        $email_settings = get_option('jobster_email_settings');
        $template = isset($email_settings['jobster_email_job_alerts_field']) 
                    ? $email_settings['jobster_email_job_alerts_field'] 
                    : '';

        if ($template != '') {
            $template = str_replace("{CANDIDATE_NAME}", $candidate_name, $template);
            $template = str_replace("{JOB_TITLE}", $job_title, $template);
            $template = str_replace("{JOB_URL}", $job_url, $template);
            $template = str_replace("{COMPANY_NAME}", $company_name, $template);

            $send = wp_mail($address, $subject, $template, $headers);
        } else {
            $message = sprintf( __('Dear %s,', 'jobster-plugin'), esc_html($candidate_name) ) . "\r\n\r\n";
            $message .= "<br/><br/>";
            if ($job_anonymous != '1') {
                $message .= sprintf(
                    __('We are excited to inform you about a new job posting for the position of <b>%s</b> at <b>%s</b>. As a valued member of our job board community, we wanted to notify you about this opportunity.', 'jobster-plugin'),
                    esc_html($job_title),
                    esc_html($company_name),
                ) . "\r\n\r\n";
            } else {
                $message .= sprintf(
                    __('We are excited to inform you about a new job posting for the position of <b>%s</b>. As a valued member of our job board community, we wanted to notify you about this opportunity.', 'jobster-plugin'),
                    esc_html($job_title),
                    esc_html($company_name),
                ) . "\r\n\r\n";
            }
            $message .= "<br/><br/>";
            $message .= sprintf(
                __('If you meet these qualifications and are interested in applying for this position, please click on the link below to visit the job posting page and submit your application:', 'jobster-plugin')
            ) . "\r\n\r\n";
            $message .= "<br/><br/>";
            $message .= '<a href="' . esc_url($job_url) . '" target="_blank">' . esc_url($job_url) . '</a>' . "\r\n\r\n";

            $send = wp_mail($address, $subject, $message, $headers);
        }
    }
endif;

if (!function_exists('jobster_job_expiration_alerts')): 
    function jobster_job_expiration_alerts() {
        $jobs_settings = get_option('jobster_jobs_settings');
        $send_alerts =  isset($jobs_settings['jobster_job_expiration_alert_field']) 
                        && $jobs_settings['jobster_job_expiration_alert_field'] == '1';
        $alerts_time = isset($jobs_settings['jobster_job_expiration_alert_time_field'])
                        ? $jobs_settings['jobster_job_expiration_alert_time_field']
                        : '';

        if ($send_alerts
            && $alerts_time != ''
            && is_numeric($alerts_time)
            && intval($alerts_time) > 0) {
            $jobs_list = jobster_get_soon_to_be_expired_jobs($alerts_time);

            foreach($jobs_list as $job) :
                $company_id = get_post_meta($job->ID, 'job_company', true);

                if (!empty($company_id)) {
                    $data = jobster_job_expiration_get_alert_data($job->ID, $company_id);

                    if (count($data) > 0) {
                        $alert_sent = get_post_meta($job->ID, 'job_valid_alert_sent', true);

                        if ($alert_sent !== '1') {
                            jobster_job_expiration_alerts_send_email($data);
                            update_post_meta($job->ID, 'job_valid_alert_sent', '1');
                        }
                    }
                }
            endforeach;
        }
    }
endif;
add_action('init', 'jobster_job_expiration_alerts');

if (!function_exists('jobster_get_soon_to_be_expired_jobs')): 
    function jobster_get_soon_to_be_expired_jobs($alerts_time) {
        $date = new DateTime();
        $date->modify('+' . $alerts_time . ' days');
        $check_date = $date->format('Y-m-d');

        $args = array(
            'posts_per_page' => -1,
            'post_type'      => 'job',
            'post_status'    => 'publish',
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key' => 'job_valid',
                    'compare' => '==',
                    'value' => $check_date,
                    'type' => 'DATE'
                ),
                array(
                    'key' => 'job_valid',
                    'compare' => '!=',
                    'value' => ''
                ),
                array(
                    'key' => 'job_valid',
                    'compare' => 'EXISTS'
                )
            )
        );

        $query = new WP_Query($args);
        wp_reset_postdata();

        return $query->get_posts();
    }
endif;

if (!function_exists('jobster_job_expiration_get_alert_data')): 
    function jobster_job_expiration_get_alert_data($job_id, $company_id) {
        $data = array();

        $job_valid     = get_post_meta($job_id, 'job_valid', true);
        $company_email = get_post_meta($company_id, 'company_email', true);

        if (!empty($company_email) && !empty($job_valid)) {
            $data['job_title']     = get_the_title($job_id);
            $data['job_url']       = get_the_permalink($job_id);
            $data['job_date']      = get_the_date($job_id);
            $data['job_valid']     = $job_valid;
            $data['company_name']  = get_the_title($company_id);
            $data['company_email'] = $company_email;
        }

        return $data;
    }
endif;

if (!function_exists('jobster_job_expiration_alerts_send_email')): 
    function jobster_job_expiration_alerts_send_email($data) {
        $subject = sprintf(__('Job Posting Expiration for [Job Title] [%s]', 'jobster-plugin'), $data['job_title']);
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_option('blogname') . '<' . get_option('admin_email') . '>',
            'Reply-To: ' . get_option('blogname') . '<' . get_option('admin_email') . '>'
        );

        $email_settings = get_option('jobster_email_settings');
        $template = isset($email_settings['jobster_email_job_expiration_alerts_field']) 
                    ? $email_settings['jobster_email_job_expiration_alerts_field'] 
                    : '';

        if ($template != '') {
            $template = str_replace(
                "{JOB_TITLE}", esc_html($data['job_title']), $template
            );
            $template = str_replace(
                "{JOB_URL}", esc_url($data['job_url']), $template
            );
            $template = str_replace(
                "{JOB_POST_DATE}",
                esc_html(date_i18n(
                    get_option('date_format'), 
                    strtotime($data['job_date'])
                )),
                $template
            );
            $template = str_replace(
                "{JOB_EXPIRY_DATE}",
                esc_html(date_i18n(
                    get_option('date_format'), 
                    strtotime($data['job_valid'])
                )),
                $template);
            $template = str_replace(
                "{COMPANY_NAME}", esc_html($data['company_name']), $template
            );

            $send = wp_mail($data['company_email'], $subject, $template, $headers);
        } else {
            $message = sprintf( __('Dear %s Recruitment Team,', 'jobster-plugin'), esc_html($data['company_name']) ) . "\r\n\r\n";
            $message .= "<br/><br/>";
            $message .= sprintf(
                            __('This is an automated notification sent to inform you that the job posting for the position of <b>%s</b>, which was listed on %s, will expire on <b>%s</b>.', 'jobster-plugin'),
                            esc_html($data['job_title']),
                            esc_html(date_i18n(get_option('date_format'), strtotime($data['job_date']))),
                            esc_html(date_i18n(get_option('date_format'), strtotime($data['job_valid'])))
                        ) . "\r\n\r\n";
            $message .= "<br/><br/>";
            $message .= esc_html__('If this position has been filled, please disregard this message. However, if the role is still open and you would like to extend the posting period, please update the job listing.', 'jobster-plugin');
            $message .= "<br/><br/>";
            $message .= '<a href="' . esc_url($data['job_url']) . '" target="_blank">' . esc_html__('See job posting details &raquo;', 'jobster-plugin') . '</a>' . "\r\n\r\n";

            $send = wp_mail($data['company_email'], $subject, $message, $headers);
        }
    }
endif;

if (!function_exists('jobster_job_expiration_alert_reset')): 
    function jobster_job_expiration_alert_reset($post_id, $post, $update) {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        if ($post->post_type !== 'job') {
            return;
        }

        $old_value = get_post_meta($post_id, 'job_valid', true);
        $new_value = isset($_POST['job_valid']) ? sanitize_text_field($_POST['job_valid']) : '';

        if ($old_value !== $new_value) {
            update_post_meta($post_id, 'job_valid_alert_sent', '0');
        }
    }
endif;
add_action('save_post_job', 'jobster_job_expiration_alert_reset', 10, 3);
?>